<?php

namespace App\Http\Controllers\api\v1;

use App\Constant\ConstantField;
use Exception;
use App\Models\User;
use App\Models\api\v1\Role;
use Illuminate\Support\Str;
use App\Models\api\v1\Order;
use App\Models\api\v1\Space;
use App\Models\api\v1\Stock;
use App\Models\api\v1\QrCodeUsers;
use App\Models\api\v1\Transaction;
use App\Models\api\v1\TransactionStatus;
use App\Models\api\v1\TransactionType;
use Illuminate\Http\Request;
use App\Models\api\v1\Device;
use App\Services\CurlService;
use App\Models\api\v1\LineItem;
use App\Models\api\v1\UserRole;
use App\Enums\OrderStatusesEnum;
use App\Services\PaymentService;
use Illuminate\Http\JsonResponse;
use App\Models\api\v1\OrderStatus;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Helpers\NotificationHelpers;
use App\Http\Controllers\Controller;
use App\Models\api\v1\Item;
use App\Models\api\v1\Wallet;
use App\Models\api\v1\UsersRolesSpace;
use App\Services\CinetPayService;
use Illuminate\Support\Facades\Validator;


class OrderController extends Controller
{

    public function userOrders(Request $request): JsonResponse
    {
        try {
            return $this->ReturnResponse(
                true,
                "User Order list",
                Order::select('orders.*')
                    ->where([['orders.created_by', request()->user()->id], ['orders.is_deleted', 0], ['orders.is_active', 1]])
                    ->OrderBy('orders.created_at', 'asc')
                    ->join('line_items', 'line_items.order_id', '=', 'orders.id')
                    ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
                    ->with(['orderStatus', 'lineItems.stock.item', 'space'])
                    ->distinct()
                    ->get()
                    ->toArray()
            );
        } catch (Exception $ex) {
            $message =  "Unable to Retrieve list of orders for User" . $ex->getMessage();
            Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [], 500);
        }
    }

    public function userOrdersWithSpace(Request $request): JsonResponse
    {
        try {
            return $this->ReturnResponse(
                true,
                "Space Order list",
                [
                    "space" => Order::select('orders.*')
                        ->where([['orders.created_by', request()->user()->id], ['orders.is_deleted', 0], ['orders.is_active', 1]])
                        ->join('line_items', 'line_items.order_id', '=', 'orders.id')
                        ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
                        ->with(['orderStatus', 'lineItems.stock.item'])
                        ->OrderBy('orders.created_at', 'desc')
                        ->first()->lineItems?->first()?->stock?->space ?? "",
                    "user_orders" =>
                    Order::select('orders.*')
                        ->where([['orders.created_by', request()->user()->id], ['orders.is_deleted', 0], ['orders.is_active', 1]])
                        ->OrderBy('orders.created_at', 'desc')
                        ->join('line_items', 'line_items.order_id', '=', 'orders.id')
                        ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
                        ->with(['orderStatus', 'lineItems.stock.item'])
                        ->get()
                        ->toArray()
                ]
            );
        } catch (Exception $ex) {
            $message =  "Unable to Retrieve list of orders for User" . $ex->getMessage();
            Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [], 500);
        }
    }


    // public function spaceOrders(Request $request, string $space_uuid): JsonResponse
    // {

    //     try {
    //         $space = Space::where([['uuid', $space_uuid], ['is_deleted', 0], ['is_active', 1]])->first();
    //         if (!$space) {
    //             return $this->ReturnResponse(false, "Space not found", [], 404);
    //         }

    //         return $this->ReturnResponse(
    //             true,
    //             "Space Order list",
    //             Order::getOrdersForSpace($space->id)
    //         );
    //     } catch (Exception $ex) {
    //         $message =  "Unable to Retrieve list of orders" . $ex->getMessage();
    //         //Log::channel('customlog')->error($message);
    //         return $this->ReturnResponse(false, $message, [], 500);
    //     }
    // }


    /**
     * Cette fonction permet d'avoir la liste des status des commandes
     * Vous pouvez avoir les status suivant l'état que vous voulez voir
     * 
     * @return mixed
     */
    public function getstatuses(Request $request): JsonResponse
    {
        $statuses           = OrderStatus::orderBy('step_number')->get()->toArray();
        $step = $request->get('after_step');
        if (isset($step)) {

            $statuses = OrderStatus::where('step_number', intval($step) + 1)->orderBy('step_number')->get()->toArray();
        }
        return $this->ReturnResponse(true, "Liste des status de commandes", $statuses);
    }

    /**
     * Cette fonction permet de mettre a jour le status d'une commande
     * 
     * @return mixed
     */
    public function changeStatus(Request $request, string $order_uuid, string $status): JsonResponse
    {
        try {
            DB::beginTransaction();
            ////check Order exists
            $order = Order::select('orders.*')->Where([['orders.uuid', $order_uuid], ['orders.is_deleted', 0], ['orders.is_active', 1]])
                ->join('line_items', 'line_items.order_id', '=', 'orders.id')
                ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
                ->first();
            if (!$order) {
                return $this->ReturnResponse(false, "Order not found", [], 404);
            }
            $currentStatus      = OrderStatus::where('id', $order->order_status_id)->first();
            $assignStatus       = OrderStatus::where('name', strtoupper($status))->first();
            if (!$assignStatus) {
                return $this->ReturnResponse(false, "Le status de commande inconnu", [], 404);
            }
            ///////get devices tokens 
            $usersInSpace = array();
            $devicesTokens = $this->getDevicesTokens($order, $usersInSpace);
            // if ($assignStatus->name == "SERVED"  || $assignStatus->name == "PAID") {
            // if ($assignStatus->name == "CONFIRMED") {
            //     $validator = Validator::make(
            //         $request->all(),
            //         [
            //             'payment_method' => 'required',
            //         ]
            //     );
            //     if ($validator->fails()) {
            //         return $this->ReturnResponse(false, $validator->errors(), [], 401);
            //     }

            //     if ($order->remaining_amount < 0) {
            //         if (request()->header('mobileId')) {
            //             NotificationHelpers::send(
            //                 [request()->header('mobileId')],
            //                 "Information sur la commande ",
            //                 "Merci de vérifier le montant de la commande N°" . $order->uuid,
            //             );
            //         }

            //         return $this->ReturnResponse(
            //             false,
            //             "Merci de vérifier le montant de la commande",
            //             [],
            //             408
            //         );
            //     }

            //     if (strtoupper($request->payment_method) == 'FLEX') {

            //         $access_token = $request->header('Authorization');
            //         $auth_header = explode(' ', $access_token);
            //         $token = $auth_header[1];

            //         $data = [

            //             "montant"       => $order->remaining_amount,
            //             "merchant"      => $userRoles?->user?->id  ?? 536,
            //             "commentaire"   => "Test de paiement marchand"

            //         ];
            //         $payment = new PaymentService(env('FLEX_API_URL_MERCHANT'), $token);
            //         $return = json_decode($payment->sendRequest('POST', $data));

            //         if ($return->success == false) {

            //             if (request()->header('mobileId')) {
            //                 NotificationHelpers::send(
            //                     [request()->header('mobileId')],
            //                     "Information sur la commande ",
            //                     "Paiement Flex a échoué : " . $return->message
            //                 );
            //             }
            //             if (!empty($devicesTokens)) {
            //                 NotificationHelpers::send(
            //                     // Device::where('user_id', $userRoles?->user?->id)->first()->device_token,
            //                     $devicesTokens,
            //                     "Information sur la commande ",
            //                     "Paiement Flex a échoué : " . $return->message
            //                 );
            //             }

            //             return $this->ReturnResponse(
            //                 false,
            //                 $return->message,
            //                 [],
            //                 408
            //             );
            //         }

            //         $message = ' .' . $return->message;
            //     }

            //     if (strtoupper($request->payment_method) == 'ONLINE') {
            //         if (empty($request->return_url)) {
            //             return $this->ReturnResponse(false, "return_url is required", [], 404);
            //         }
            //         $order->fees = 100 + $order->remaining_amount * 0.01;
            //         $order->save();
            //         /////creation de la transaction
            //         $trans_status = TransactionStatus::where('name', 'INIT')->first();
            //         $wallet = Wallet::where('name', 'CINETPAY')->first();
            //         $trans_type = TransactionType::where('name', "ONLINE")->first();
            //         $data = [
            //             "trans_status" => $trans_status->id,
            //             "wallet" => $wallet->id,
            //             "trans_type" => $trans_type->id,
            //             "request" => "",
            //             "response" => "",
            //             "callback_response" => "",
            //             "created_by" => $request->user()->id,
            //         ];
            //         $transaction = $this->createTransactionFromorder($order, $data);
            //         //////////////////////////////////////////////////////////////////////
            //         $data = [
            //             "amount"       => $order->remaining_amount + $order->fees,
            //             "description"   => "Paiement Marchand",
            //             "libelle"   => "Paiement Marchand",
            //             "transaction_id"   => $transaction->uuid,
            //             "name"   => $order->customer->lastname . '  ' . $order->customer->firstname,
            //             "tel"   => $order->customer->username,
            //             "return_url"   => $request->return_url,
            //         ];
            //         Log::info('test', ["data" => $data]);
            //         try {
            //             $payment = new CinetPayService();
            //             $urlPay = $payment->generatePaymentLink($data);
            //             Log::info('paiement ok', ["data" => $urlPay]);
            //             if ($urlPay["code"] == '201') {
            //                 $url = $urlPay["data"]["payment_url"];
            //                 $data = [
            //                     "request" => $urlPay['request'],
            //                     "response" => $urlPay['response'],
            //                     "callback_response" => "",
            //                 ];
            //                 Transaction::where('uuid', $transaction->uuid)->update($data);
            //             } else {
            //                 $message = "Une erreur s'est produite";
            //                 //  $userid = $userRoles?->user?->id;
            //                 return $this->paymentFail($message, $devicesTokens);
            //             }
            //         } catch (\Exception $e) {
            //             $message = $e->getMessage();
            //             Log::error($message);
            //             //$userid = $userRoles?->user?->id;
            //             return $this->paymentFail($message, $devicesTokens);
            //         }
            //     }

            //     $this->updateStock($order, "CONFIRMED");
            //     $order->order_status_id = $assignStatus->id;
            //     $order->updated_at      = now();
            //     $order->updated_by      = $request->user()->id;
            //     $order->payment_method    = $request->payment_method;
            //     $order->save();

            //     DB::commit();

            //     if (request()->header('mobileId')) {
            //         NotificationHelpers::send(
            //             [request()->header('mobileId')],
            //             "Information sur la commande ",
            //             "Votre paiement de la commande N°" . $order->uuid . " est en attente de paiement"
            //         );
            //     }
            //     if (!empty($devicesTokens)) {
            //         NotificationHelpers::send(
            //             // Device::where('user_id', $userRoles?->user?->id)->first()->device_token,
            //             $devicesTokens,
            //             "Information sur la commande ",
            //             "Le paiement de la commande N°" . $order->uuid . "  a été initié et en attente de paiement"
            //         );
            //     }
            //     return $this->ReturnResponse(
            //         true,
            //         "Votre paiement de la commande N°" . $order->order_number . " est en attente de paiement",
            //         ["url" => $url, "order" => $order]
            //     );
            // }

            //////Processing order with status Confirmed
            if ($assignStatus->name == "CONFIRMED") {
                $response = $this->processOrderWithStatusConfirm($request, $order, $assignStatus, $devicesTokens);
                if (!$response['status']) {
                    return $this->ReturnResponse(false, $response['message'], $response['data'], $response['status']);
                }
            }

            // if ($assignStatus->name == "PAID") {
            //     $order->fees = 100 + $order->remaining_amount * 0.01;
            //     $order->save();
            //     $trans_status = TransactionStatus::where('name', 'APPROVED')->first();
            //     $trans_type = TransactionType::where('name', "CASH")->first();
            //     $data = [
            //         "trans_status" => $trans_status->id,
            //         "wallet" => null,
            //         "trans_type" => $trans_type->id,
            //         "request" => "",
            //         "response" => "",
            //         "callback_response" => "",
            //         "created_by" => $request->user()->id,
            //     ];
            //     $this->createTransactionFromorder($order, $data);
            //     $userRoles = UserRole::whereIn('id', $userInSpace)->where('role_id', 2)->get();
            //     foreach ($userRoles as $userRole) {
            //         $device =  Device::where('user_id', $userRole?->user?->id)->first();
            //         if (!empty($device)) {
            //             array_push($devicesTokens, $device->device_token);
            //         }
            //     }
            // }

            //////Processing order with status Paid
            if ($assignStatus->name == "PAID") {
                $this->processOrderWithStatusPaid($request, $order, $usersInSpace);
                $order->paid      = 1;
            }

            ////////Processing order with status Cancel
            if ($assignStatus->name == "CANCELED") {
                $this->updateStock($order, "CANCELED");
            }

            $order->order_status_id = $assignStatus->id;
            $order->updated_at      = now();
            $order->updated_by      = $request->user()->id;
            $order->save();
            if (request()->header('mobileId')) {
                // NotificationHelpers::send(
                //     [request()->header('mobileId')],
                //     "Information sur la commande ",
                //     "Le statut de la commande " . $order->order_number  . " est passé de " . $currentStatus->name . " à " . $assignStatus->name,
                // );
            }
            if (!empty($devicesTokens)) {
                // NotificationHelpers::send(
                //     // Device::where('user_id', $userRoles?->user?->id)->first()->device_token,
                //     $devicesTokens,
                //     "Information sur la commande ",
                //     "Le statut de la commande " . $order->order_number  . "  est passé de " . $currentStatus->name . " à " . $assignStatus->name,
                // );
            }
            DB::commit();
            return $this->ReturnResponse(
                true,
                "Le statut de la commande est passé de " . $currentStatus->name . " à " . $assignStatus->name,
                [$order]
            );
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossible de modifier cette commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [], 500);
        }
    }



    /**
     * get Devices Tokens
     *
     * @param  mixed $order
     * @param  mixed $userInSpace
     * @return array
     */
    private function getDevicesTokens(Order $order, array &$userInSpace): array
    {
        $userInSpace = UsersRolesSpace::where('space_id', $order->space_id)->get()->pluck('user_role_id')->toArray();
        $userRoles = UserRole::whereIn('id', $userInSpace)->where('role_id', 3)->get();
        $devicesTokens = array();
        foreach ($userRoles as $userRole) {
            $device =  Device::where('user_id', $userRole?->user?->id)->first();
            if (!empty($device)) {
                array_push($devicesTokens, $device->device_token);
            }
        }
        return $devicesTokens;
    }




    /**
     * process Order With Status Confirm
     *
     * @param  mixed $request
     * @param  mixed $order
     * @param  mixed $assignStatus
     * @param  mixed $deviceTokens
     * @return array
     */
    private function processOrderWithStatusConfirm(Request $request, Order $order, OrderStatus $assignStatus, array $deviceTokens): array
    {
        $validator = Validator::make(
            $request->all(),
            [
                'payment_method' => 'required',
            ]
        );
        if ($validator->fails()) {
            return  ["success" => false, "message" => $validator->errors(), "status" => 400];
        }
        if ($order->remaining_amount < 0) {
            if (request()->header('mobileId')) {
                // NotificationHelpers::send(
                //     [request()->header('mobileId')],
                //     "Information sur la commande ",
                //     "Merci de vérifier le montant de la commande N°" . $order->order_number,
                // );
            }
            return  ["success" => false, "message" => "Merci de vérifier le montant de la commande", "status" => 408];
        }
        if (strtoupper($request->payment_method) == 'FLEX') {
            $paymentFlex = $this->processOrderWithStatusConfirmFlex($request, $order, $deviceTokens);
            return $paymentFlex;
        }
        $generateUrl = [];
        if (strtoupper($request->payment_method) == 'ONLINE') {
            $generateUrl =  $this->processOrderWithStatusConfirmOnline($request, $order, $deviceTokens);
            if (!$generateUrl['success']) {
                return $generateUrl;
            }
        }
        //////Update Stock and notify users
        // $return = [];
        $data = !empty($generateUrl) ? $generateUrl['data'] : '';
        $response = $this->updateStockAndNotifyUsers($request, $order, $assignStatus, $data);

        return $response;
    }

    /**
     * process Order With Status Paid
     *
     * @param  mixed $request
     * @param  mixed $order
     * @param  mixed $userInSpace
     * @return void
     */
    private function processOrderWithStatusPaid(Request $request, Order $order, array $userInSpace)
    {
        $order->fees = 100 + $order->remaining_amount * 0.01;
        $order->save();
        $trans_status = TransactionStatus::where('name', 'APPROVED')->first();
        $trans_type = TransactionType::where('name', "CASH")->first();
        $data = [
            "trans_status" => $trans_status->id,
            "wallet" => null,
            "trans_type" => $trans_type->id,
            "request" => "",
            "response" => "",
            "callback_response" => "",
            "created_by" => $request->user()->id,
        ];
        $this->createTransactionFromorder($order, $data);
        // $userRoles = UserRole::whereIn('id', $userInSpace)->where('role_id', 2)->get();
        // foreach ($userRoles as $userRole) {
        //     $device =  Device::where('user_id', $userRole?->user?->id)->first();
        //     if (!empty($device)) {
        //         array_push($devicesTokens, $device->device_token);
        //     }
        // }
    }

    /**
     * update Stock And Notify Users
     *
     * @param  mixed $request
     * @param  mixed $order
     * @param  mixed $assignStatus
     * @param  mixed $url
     * @return array
     */
    private function updateStockAndNotifyUsers(Request $request, Order $order, OrderStatus $assignStatus, String $url): array
    {
        $this->updateStock($order, "CONFIRMED");
        $order->order_status_id = $assignStatus->id;
        $order->updated_at      = now();
        $order->updated_by      = $request->user()->id;
        $order->payment_method    = $request->payment_method;
        $order->save();
        DB::commit();
        if (request()->header('mobileId')) {
            // NotificationHelpers::send(
            //     [request()->header('mobileId')],
            //     "Information sur la commande ",
            //     "Votre paiement de la commande N°" . $order->order_number . " est en attente de paiement"
            // );
        }
        // if (!empty($devicesTokens)) {
        //     NotificationHelpers::send(
        //         // Device::where('user_id', $userRoles?->user?->id)->first()->device_token,
        //         $devicesTokens,
        //         "Information sur la commande ",
        //         "Le paiement de la commande N°" . $order->order_number . "  a été initié et en attente de paiement"
        //     );
        // }
        $data = ["url" => $url, "order" => $order];
        $message = "Votre paiement de la commande N°" . $order->order_number . " est en attente de paiement";
        return ["success" => true, "message" => $message, "data" => $data, "status" => 200];
    }


    /**
     * process Order With Status Confirm Flex
     *
     * @param  mixed $request
     * @param  mixed $order
     * @param  mixed $deviceTokens
     * @return array
     */
    private function processOrderWithStatusConfirmFlex(Request $request, Order $order, array $deviceTokens): array
    {
        $access_token = $request->header('Authorization');
        $auth_header = explode(' ', $access_token);
        $token = $auth_header[1];
        $data = [
            "montant"       => $order->remaining_amount,
            "merchant"      => $userRoles?->user?->id  ?? 536,
            "commentaire"   => "Test de paiement marchand"
        ];
        $payment = new PaymentService(env('FLEX_API_URL_MERCHANT'), $token);
        $return = json_decode($payment->sendRequest('POST', $data));
        if ($return->success == false) {
            if (request()->header('mobileId')) {
                // NotificationHelpers::send(
                //     [request()->header('mobileId')],
                //     "Information sur la commande ",
                //     "Paiement Flex a échoué : " . $return->message
                // );
            }
            if (!empty($devicesTokens)) {
                // NotificationHelpers::send(
                //     // Device::where('user_id', $userRoles?->user?->id)->first()->device_token,
                //     $devicesTokens,
                //     "Information sur la commande ",
                //     "Paiement Flex a échoué : " . $return->message
                // );
            }
            return ['success' => false, "message" => $return->message, "status" => 408];
        }
        return ['success' => true, "message" => $return->message, "status" => 200];
    }

    /**
     * process Order With Status Confirm Online
     *
     * @param  mixed $request
     * @param  mixed $order
     * @param  mixed $deviceTokens
     * @return void
     */
    private function processOrderWithStatusConfirmOnline(Request $request, Order $order, array $deviceTokens)
    {
        if (empty($request->return_url)) {
            return ['success' => false, "message" => "return_url is required", "status" => 404];
        }
        $order->fees = 100 + $order->remaining_amount * 0.01;
        $order->save();
        /////creation de la transaction
        $trans_status = TransactionStatus::where('name', 'INIT')->first();
        $wallet = Wallet::where('name', 'CINETPAY')->first();
        $trans_type = TransactionType::where('name', "ONLINE")->first();
        $data = [
            "trans_status" => $trans_status->id,
            "wallet" => $wallet->id,
            "trans_type" => $trans_type->id,
            "request" => "",
            "response" => "",
            "callback_response" => "",
            "created_by" => $request->user()->id,
        ];
        $transaction = $this->createTransactionFromorder($order, $data);
        //////////////////////////////////////////////////////////////////////
        $data = [
            "amount"       => $order->remaining_amount + $order->fees,
            "description"   => "Paiement Marchand",
            "libelle"   => "Paiement Marchand",
            "transaction_id"   => $transaction->uuid,
            "name"   => $order->customer->lastname . '  ' . $order->customer->firstname,
            "tel"   => $order->customer->username,
            "return_url"   => $request->return_url,
        ];
        Log::info('test', ["data" => $data]);
        try {
            $payment = new CinetPayService();
            $urlPay = $payment->generatePaymentLink($data);
            Log::info('paiement ok', ["data" => $urlPay]);
            if ($urlPay["code"] == '201') {
                $url = $urlPay["data"]["payment_url"];
                $data = [
                    "request" => $urlPay['request'],
                    "response" => $urlPay['response'],
                    "callback_response" => "",
                ];
                Transaction::where('uuid', $transaction->uuid)->update($data);
            } else {
                $message = "Une erreur s'est produite";
                //  $userid = $userRoles?->user?->id;
                return $this->paymentFail($message, $deviceTokens);
            }
            return ["success" => true, "message" => "", "data" => $url, "status" => 200];
        } catch (\Exception $e) {
            $message = $e->getMessage();
            Log::error($message);
            return $this->paymentFail($message, $deviceTokens);
        }
    }

    /****
     * Create transaction when order is created with Paid status or online payment
     */
    private function createTransactionFromorder($order, array $data)
    {

        $trans = [
            "uuid" => Str::uuid()->toString(),
            "amount_paid" => $order->remaining_amount + $order->fees,
            "order_id"    => $order->id,
            "transaction_status_id" => $data['trans_status'],
            "wallet_id" => $data['wallet'],
            "request" => json_encode($data['request']) ?? "",
            "response" => json_encode($data['response']) ?? "",
            "callback_response" => json_encode($data['callback_response']) ?? "",
            "transaction_type_id" => $data['trans_type'],
            "created_at" => now(),
            "created_by" => $data['created_by']
        ];
        $transaction = Transaction::updateOrCreate(["order_id" => $order->id, "transaction_status_id" => $data['trans_status']], $trans);
        return $transaction;
    }

    /*****
     * Update Stock from mouvement
     */
    /*****
     * Update Stock from mouvement
     */
    private function updateStock($order, $status, $merchantId = null)
    {
        if ($order->order_status_id == OrderStatus::where('name', $status)->first()?->id) {
            return;
        }
        try {
            \DB::beginTransaction();
            $lineItems = LineItem::where('order_id', $order->id)->get();
            if (empty($lineItems)) {
                return;
            }
            $statusPaidId = OrderStatus::where('name', "PAID")->first()?->id;
            $statusServedId = OrderStatus::where('name', "SERVED")->first()?->id;
            foreach ($lineItems as $line) {
                $stock = Stock::where('id', $line->stock_id)->first();
                $updateStock = 0;
                if (!empty($stock)) {
                    if ($status == "CANCELED") {
                        $updateStock = $stock->quantity + $line->quantity;
                    } else if ($status == "CONFIRMED") {
                        $updateStock = $stock->quantity -  $line->quantity;
                    }
                    $stock->quantity = $updateStock;
                    $stock->save();
                }
            }
            if ($order->order_status_id == $statusPaidId) {
                $data = $order->toArray();
                $data["order_status_id"] = $statusServedId;
                $data["uuid"] = \Str::uuid()->tostring();
                $data['amount_due'] = $data['remaining_amount'] =  (-1) * $order->amount_due;
                $data['created_at'] = now();
                $data['created_by'] = $merchantId;
                $ordercreate = Order::create($data);
                if ($order->payment_method == "online" || $order->payment_method == "mobile") {
                    ////Appel Service Cinetpay 
                }
            }
            \DB::commit();
        } catch (\Exception $e) {
            \DB::rollback();
            dd($e->getMessage());
        }
    }


    /**
     * Cette fonction retourne les commandes d'un client au scan de son QrCode
     * 
     * @return mixed
     */
    public function getClientByQrCode($num_card, $space_uuid)
    {
        if (empty($num_card) || empty($space_uuid)) {
            return $this->ReturnResponse(false, "Empty Space or QrCode ", [], 401);
        }
        $client = QrCodeUsers::where('num_carte', $num_card)->with('cartes_users')->first();
        if (empty($client)) {
            return $this->ReturnResponse(false, "Unable to Retrieve qr code data", [], 400);
        }
        $space = Space::where('uuid', $space_uuid)->first();
        if (empty($space)) {
            return $this->ReturnResponse(false, "Unable to Retrieve space data", [], 400);
        }
        $ordersStatus = OrderStatus::where('name', '!=', 'CART')
            ->where('name', '!=', "CANCELED")
            ->where('name', '!=', "SERVED")
            ->get()
            ->pluck('id')
            ->toArray();
        if (empty($client->cartes_users->id)) {
            return $this->ReturnResponse(false, "Unable to Retrieve client data ", [], 400);
        }
        $orders = Order::select('orders.*')
            ->where([['orders.created_by', $client->cartes_users->id], ['orders.is_deleted', 0], ['orders.is_active', 1], ['orders.space_id', $space->id]])
            ->whereIn("order_status_id", $ordersStatus)
            ->OrderBy('orders.created_at', 'desc')
            ->join('line_items', 'line_items.order_id', '=', 'orders.id')
            ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
            ->with(['orderStatus', 'lineItems.stock.item'])
            ->distinct('orders.id')
            ->get()
            ->toArray();
        $customer["nom"] = $client->cartes_users->firstname;
        $customer["prenom"] = $client->cartes_users->lastname;
        $data = ['orders' => $orders, "customer" => $customer];
        return $this->ReturnResponse(true, "Liste des Commandes Clients", $data, 200);
    }


    public function show(Request $request, string $uuid): JsonResponse
    {
        $order = Order::Where([['uuid', $uuid], ['is_deleted', 0], ['is_active', 1]])
            ->with(['orderStatus', 'lineItems.stock.item'])->first();
        if (!$order) {
            return $this->ReturnResponse(false, "Order not found", [], 404);
        }

        return $this->ReturnResponse(
            true,
            "Order Details",
            [$order]
        );
    }

    public function show_space(Request $request, string $space_uuid, string $uuid): JsonResponse
    {
        return $this->show($request, $uuid);
    }

    public function create(Request $request): JsonResponse
    {

        $validator = Validator::make(
            $request->all(),
            [
                'space_id'                 => 'required|exists:spaces,id',
                'line_items'               => 'required|array',
                'line_items.*.stock_id'    => 'required|exists:stocks,id',
                'line_items.*.quantity'    => 'required|min:1',
                "table_number"             => "string",
            ]
        );

        //si erreur
        if ($validator->fails()) {
            return $this->ReturnResponse(false, $validator->errors(), [], 400);
        }
        try {
            $orders = Order::where([
                ['created_by', $request->user()->id],
                ['space_id', $request->space_id],
                ['order_status_id', OrderStatus::where('name', 'CONFIRMED')->first()->id]
            ])->get()->toArray();
            if (count($orders) >= 3) {
                return $this->ReturnResponse(false, "Vous avez plus de 3 commandes en attente de paiement", [], 400);
            }

            $order = Order::where([
                ['created_by', $request->user()->id],
                ['order_status_id', OrderStatus::where('name', 'CART')->first()->id]
            ])->with(['orderStatus', 'lineItems'])->first();


            $amount_due = 0;
            if ($order) {
                $order->order_status_id =  OrderStatus::where('name', 'CANCELED')->first()->id;
                $order->save();
            }
            $order = Order::create([
                "order_status_id"       => 1,
                "space_id"              => $request->space_id,
                "uuid"                  => Str::uuid(),
                "amount_due"            => $amount_due,
                "table_number"          => $request->table_number,
                "remaining_amount"      => $amount_due,
                "created_by"            => $request->user()->id,
                "updated_by"            => $request->user()->id,
                "payment_method"           => $request->payment_method,
                "is_active"             => 1
            ]);

            // if (!$order) {
            //     $order = Order::create([
            //         "order_status_id"       => 1,
            //         "uuid"                  => Str::uuid(),
            //         "amount_due"            => $amount_due,
            //         "table_number"          => $request->table_number,
            //         "remaining_amount"      => $amount_due,
            //         "created_by"            => $request->user()->id,
            //         "updated_by"            => $request->user()->id,
            //         "payment_method"           => $request->payment_method,
            //         "is_active"             => 1
            //     ]);
            // } else {
            //     return $this->ReturnResponse(false, "Il existe déjà un panier. Merci de mettre à jour le panier", [], 400);
            // }


            $stock_id_list   = [];
            $list_line_items = [];
            foreach ($request->line_items as $key => $sline) {
                $stock_id_list[$request->line_items[$key]["stock_id"]] = $sline["quantity"];
            }


            DB::beginTransaction();
            $stocks = Stock::where('space_id', $request->space_id)->whereIn('id', array_keys($stock_id_list))->get();


            if (count($stocks) != count($stock_id_list)) {
                return $this->ReturnResponse(false, "Certains articles n'appartiennent pas à cet espace", [], 404);
            }
            $amount_due = 0;

            foreach ($stocks as $key => $stock) {
                $line = [
                    "stock_id"      => $stock->id,
                    "order_id"      => $order->id,
                    "quantity"      => $stock_id_list[$stock->id],
                    "created_by"    => $request->user()->id,
                    "updated_by"    => $request->user()->id,
                ];
                $amount_due += $stock->price_after_sale *  $stock_id_list[$stock->id];
                $list_line_items[] = $line;
            }


            //dd($order->with(['lineItems','orderStatus']));
            LineItem::insert($list_line_items); // inserrton par lot de 10 dans la boucle 
            //$order->lineItems()->insert($list_line_items); // inserrton par lot de 10 dans la boucle 
            $order->update([
                "order_number"      => ConstantField::ORDER_NUMBER + $order->id,
                "amount_due"        => $amount_due,
                "remaining_amount"  => $amount_due,
                "payment_method"    => $request->payment_method,

            ]);
            /*"*/
            DB::commit();

            $order = Order::where('id', $order->id)->with(['orderStatus', 'lineItems'])->first();


            return $this->ReturnResponse(true, "La commande a été enregistrée", [$order]);
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossible de mettre a jour la commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            Log::error($message);
            return $this->ReturnResponse(false, "An error has occurred", [], 500);
        }
    }

    /**
     * Get recents Orders
     *
     * @param  mixed $limit
     * @return void
     */
    public function recentOrders(Request $request, $limit = 5)
    {
        $orders = Order::Where([['is_deleted', 0], ['is_active', 1], ['created_by', $request->user()->id]])
            ->with(['orderStatus', 'lineItems.stock.item'])
            ->orderBy('id', 'desc')
            ->limit($limit)->get();
        $items = array();
        $itemIds = array();
        if (!empty($orders)) {
            foreach ($orders as $order) {
                if (!empty($order->lineItems)) {
                    foreach ($order->lineItems as $lineItem) {
                        $item = $lineItem?->stock?->item;
                        if (!empty($item) && !in_array($item->id, $itemIds)) {
                            $item = $item->toArray();
                            $item['quantity_ordered'] = $lineItem->quantity;
                            array_push($items, $item);
                            array_push($itemIds, $item['id']);
                        }
                    }
                }
            }
        }
        return $this->ReturnResponse(
            true,
            "Recents orders",
            $items
        );
    }

    //////////////////////////////////////////////////// order wholesaler //////////////////////////////////////
    /**
     * createSpaceOrder
     * Create an order for a space with a wholesaler
     * @param  mixed $uuid
     * @param  mixed $request
     * @return JsonResponse
     */
    public function createCustomerSpaceOrder(string $uuid, Request $request): JsonResponse
    {
        $validator = Validator::make(
            $request->all(),
            [
                'wholesaler_id'                 => 'required|exists:spaces,uuid',
                'line_items'               => 'required|array',
                'line_items.*.stock_id'    => 'required|exists:stocks,id',
                'line_items.*.quantity'    => 'required|min:1',
                // "table_number"             => "string",
            ]
        );

        //si erreur
        if ($validator->fails()) {
            return $this->ReturnResponse(false, $validator->errors(), [], 400);
        }
        /////get space 
        $space = Space::where('uuid', $request->wholesaler_id)->first();
        if (empty($space)) {
            return $this->ReturnResponse(false, "wholesaler not found", [], 404);
        }
        $customer_space = Space::where('uuid', $uuid)->first();
        if (empty($space)) {
            return $this->ReturnResponse(false, "wholesaler not found", [], 404);
        }
        if (empty($customer_space)) {
            return $this->ReturnResponse(false, "customer space not found", [], 404);
        }
        try {
            $orders = Order::where([
                ['created_by', $request->user()->id],
                ['space_id', $request->wholesaler_id],
                ['order_status_id', OrderStatus::where('name', 'CONFIRMED')->first()->id]
            ])->get()->toArray();
            if (count($orders) >= 3) {
                return $this->ReturnResponse(false, "Vous avez plus de 3 commandes en attente de paiement", [], 400);
            }

            $order = Order::where([
                ['created_by', $request->user()->id],
                ['order_status_id', OrderStatus::where('name', 'CART')->first()->id]
            ])->with(['orderStatus', 'lineItems'])->first();


            $amount_due = 0;
            if ($order) {
                $order->order_status_id =  OrderStatus::where('name', 'CANCELED')->first()->id;
                $order->save();
            }
            $order = Order::create([
                "order_status_id" => OrderStatus::where('name', 'CONFIRMED')->first()->id,
                "space_id"              => $space->id,
                "uuid"                  => Str::uuid(),
                "amount_due"            => $amount_due,
                "table_number"          => $request->table_number,
                "customer_space_id"          => $customer_space->id,
                "remaining_amount"      => $amount_due,
                "fees"                  => $request->fees ?? 0,
                "created_by"            => $request->user()->id,
                "updated_by"            => $request->user()->id,
                "payment_method"           => $request->payment_method,
                "is_active"             => 1,
                "day"                   => $request->day ?? null,
                "slots"                 => $request->slots ?? null,
            ]);
            $stock_id_list   = [];
            $list_line_items = [];
            foreach ($request->line_items as $key => $sline) {
                $stock_id_list[$request->line_items[$key]["stock_id"]] = $sline["quantity"];
            }


            DB::beginTransaction();
            $stocks = Stock::where('space_id', $space->id)->whereIn('id', array_keys($stock_id_list))->get();


            if (count($stocks) != count($stock_id_list)) {
                return $this->ReturnResponse(false, "Certains articles n'appartiennent pas à cet espace", [], 404);
            }
            $amount_due = 0;

            foreach ($stocks as $key => $stock) {
                $line = [
                    "stock_id"      => $stock->id,
                    "order_id"      => $order->id,
                    "quantity"      => $stock_id_list[$stock->id],
                    "created_by"    => $request->user()->id,
                    "updated_by"    => $request->user()->id,
                ];
                $amount_due += $stock->price_after_sale *  $stock_id_list[$stock->id];
                $list_line_items[] = $line;
            }
            LineItem::insert($list_line_items); // inserrton par lot de 10 dans la boucle 
            //$order->lineItems()->insert($list_line_items); // inserrton par lot de 10 dans la boucle 
            $order->update([
                "order_number"      => ConstantField::ORDER_NUMBER + $order->id,
                "amount_due"        => $amount_due,
                "remaining_amount"  => $amount_due,
                "payment_method"    => $request->payment_method,

            ]);
            /*"*/
            DB::commit();
            $order = Order::where('id', $order->id)->with(['orderStatus', 'lineItems'])->first();
            return $this->ReturnResponse(true, "La commande a été enregistrée", [$order]);
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossible de mettre a jour la commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            Log::error($message);
            return $this->ReturnResponse(false, "An error has occurred", [], 500);
        }
    }

    /**
     * getOrdersSpaceToWholeslaer
     *Collect orders for a space sent to a wholesaler
     * @param  mixed $request
     * @param  mixed $space_uuid
     * @param  mixed $wholesaler_uuid
     * @return JsonResponse
     */
    public function getOrdersSpaceToWholesaler(string $space_uuid, string $wholesaler_uuid, String $status = null): JsonResponse
    {
        if (empty($space_uuid)) {
            return $this->ReturnResponse(false, "Space not found", [], 404);
        }
        if (empty($wholesaler_uuid)) {
            return $this->ReturnResponse(false, "Wholesaler not found", [], 404);
        }
        try {
            $space = Space::where([['uuid', $space_uuid], ['is_deleted', 0], ['is_active', 1]])->first();
            if (!$space) {
                return $this->ReturnResponse(false, "Space not found", [], 404);
            }
            $wholesaler = Space::where([['uuid', $wholesaler_uuid], ['is_deleted', 0], ['is_active', 1]])->first();
            if (!$wholesaler) {
                return $this->ReturnResponse(false, "Wholesaler not found", [], 404);
            }


            return $this->ReturnResponse(
                true,
                "Space Order list",
                Order::getOrdersForCustomerSpace($space->id, $wholesaler->id, $status)
            );
        } catch (Exception $ex) {
            $message =  "Unable to Retrieve list of orders" . $ex->getMessage();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [], 500);
        }
    }


    ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    public function assignOrderToWaiter(Request $request, string $space_uuid, string $order_uuid): JsonResponse
    {
        $validator = Validator::make(
            $request->all(),
            [
                'prepaid'   => 'required|string'
            ]
        );
        //si erreur
        if ($validator->fails()) {
            return $this->ReturnResponse(false, $validator->errors(), [], 400);
        }
        $prepaid = $request->prepaid;
        if($prepaid){
            $order = Order::where([
                ['uuid', $order_uuid],
                ['is_deleted', 0],
                // ['paid', 1]
            ])->first();
        } else {
            $order = Order::where([
                ['uuid', $order_uuid],
                ['is_deleted', 0],
            ])->first();
        }

        if (!$order) {
            return $this->ReturnResponse(false, "Order not found", [], 404);
        }
        if (!empty($order->server_id)) {
            return $this->ReturnResponse(false, "Order already assigned to waiter", [], 200);
        }

        try {
            DB::beginTransaction();
            $order->server_id   = $request->user()->id;
            $order->updated_at  = now();
            $order->updated_by  = $request->user()->id;

            $order->save();

            $userInSpace = UsersRolesSpace::where('space_id', $order->space_id)->get()->pluck('user_role_id');
            $userRoles = UserRole::whereIn('id', $userInSpace)->where('role_id', 2)->get();
            $devicesTokens = array();
            foreach ($userRoles as $userRole) {
                $device =  Device::where('user_id', $userRole?->user?->id)->first();
                if (!empty($device)) {
                    array_push($devicesTokens, $device->device_token);
                }
            }

            if (request()->header('mobileId')) {
                // NotificationHelpers::sendNotification(
                //     [request()->header('mobileId')],
                //     "Information sur la commande ",
                //     "Votre commande est en cours de traitement"
                // );
            }

            if (!empty($devicesTokens)) {
                // NotificationHelpers::sendNotification(
                //     [$devicesTokens],
                //     "Information sur la commande ",
                //     "La commande N°" . $order->order_number . " est en cours de traitement",
                //     ["Action" => "REMOVE", "order_mumber" => $order->order_number]
                // );
            }

            DB::commit();
            return $this->ReturnResponse(true, "Order Updated", [$order]);
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossibme de mettre a jour la commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [$order], 500);
        }
    }

    public function assignOrderByWaiter(Request $request, string $space_uuid, string $order_uuid, string $waiter_uuid): JsonResponse
    {
        $order = Order::where([
            ['uuid', $order_uuid],
            ['is_deleted', 0],
            ['order_status_id', '!=', OrderStatus::where('name', 'CART')->first()->id]
        ])->first();

        if (!$order) {
            return $this->ReturnResponse(false, "Order not found", [], 404);
        }

        try {
            DB::beginTransaction();
            $order->server_id   = User::where('uuid', $waiter_uuid)->first()->id;
            $order->updated_at  = now();
            $order->updated_by  = $request->user()->id;

            $order->save();
            DB::commit();
            return $this->ReturnResponse(true, "Order assigned", [$order]);
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossibme de mettre a jour la commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [$order], 500);
        }
    }


    public function OrderToWaiter(Request $request, string $space_uuid, string $waiter_uuid): JsonResponse
    {
        $waiter = User::where([
            ['uuid', $waiter_uuid],
            ['blocked', 0],
            ['is_active', 1],
            ['is_deleted', 0]
        ])->first();

        if (!$waiter) {
            return $this->ReturnResponse(false, "Waiter not found", [], 404);
        }
        $waiter_role = UserRole::where([
            ['is_active', 1],
            ['is_deleted', 0],
            ['user_id', $waiter->id],
            ['role_id', Role::where('name', 'Serveur')->first()->id]
        ])->first();
        if (!$waiter_role) {
            return $this->ReturnResponse(false, "USer was not waiterw", [], 404);
        }

        $space = Space::where('uuid', $space_uuid)->first();
        if (!$space) {
            return $this->ReturnResponse(false, "Space not found", [], 404);
        }

        $waiter_in_space = UsersRolesSpace::where([
            ['space_id', $space->id],
            ['user_role_id', $waiter_role->id]
        ])->first();
        if (!$waiter_in_space) {
            return $this->ReturnResponse(false, "user not waiter in this space", [], 404);
        }

        return $this->ReturnResponse(true, "Order to waiter", [
            "Assignées" => Order::where('server_id', $waiter->id)->get(),
            "Attente" => Order::where([['server_id', $waiter->id], ['order_status_id', OrderStatus::where('name', 'CONFIRMED')->first()->id]])->get(),
            "Livrées" => Order::where([['server_id', $waiter->id], ['order_status_id', OrderStatus::where('name', 'SERVED')->first()->id]])->get(),
        ], 200);
    }


    // public function OrderToSpace(Request $request, string $space_uuid): JsonResponse
    public function spaceOrders(Request $request, string $space_uuid): JsonResponse
    {
        $space = Space::where('uuid', $space_uuid)->first();
        if (!$space) {
            return $this->ReturnResponse(false, "Space not found", [], 404);
        }
        $value      = $request->value;
        $qrcode     = $request->qrcode;
        $prepaid    = isset($request->prepaid) ? $request->prepaid : 1;
        $query = null;
        if (!empty($value)) {
            if (strlen($value) == 10) {
                $query = "phone";
            } else if (substr($value, 0, 3) === "000") {
                $query = "ordertable";
            } else {
                $query = "ordernumber";
            }
        }
        $rolename = $request->rolename;
        $serverId  = null;
        $search = array();
        if (!empty($qrcode)) {
            $client = QrCodeUsers::where('num_carte', $qrcode)->with('cartes_users')->first();
            if (empty($client)) {
                return $this->ReturnResponse(false, "Unable to Retrieve qr code data", [], 400);
            }
            $search[] = ["orders.created_by", $client->cartes_users->id];
        }
        if ($rolename != "Merchant") {
            $serverId = $request->user()->id;
        }
        if (!empty($query) && !empty($value)) {
            if ($query == "phone") {
                if (!(substr($value, 0, 4) == "+225")) {
                    $value = "+225" . $value;
                }
                $user = User::where('username', $value)->first();
                if (!empty($user)) {
                    $search[] = ["orders.created_by", $user->id];
                } else {
                    return $this->ReturnResponse(true, "Order list by order status", [
                        'to_pay' => [],
                        'to_treat' => [],
                        'to_deliver' => [],
                        'history' => []
                    ], 200);
                }
            } else if ($query == "ordernumber") {
                $search[] = ["orders.order_number", $value];
            } else if ($query == "ordertable") {
                $search[] = ["orders.table_number", $value];
            }
        }
        $orderStatuses = [OrderStatus::where('name', 'SERVED')->first()->id, OrderStatus::where('name', 'CANCELED')->first()->id];
        if($prepaid){
            $search_unpaid  = empty($search) ? [["orders.paid", 0]] : [["orders.paid", 0], $search];
            $search_paid    = empty($search) ? [["orders.paid", 1]] : [["orders.paid", 1], $search];
            return $this->ReturnResponse(true, "Order list by order status", [
                'to_pay' => Order::getOrdersForSpaceByStatus($space->id, OrderStatus::where('name', 'CONFIRMED')->first()->id, $search_unpaid),
                'to_treat' => Order::getOrdersForSpaceByStatus($space->id, OrderStatus::whereIn('name', array('PAID', 'CONFIRMED'))->first()->id, $search_paid),
                'to_deliver' => Order::getOrdersForSpaceToDeliver($space->id, OrderStatus::whereIn('name', array('PAID', 'CONFIRMED'))->first()->id, $serverId, $search_paid),
                'history' => Order::getHistoryOrdersForSpace($space->id, $orderStatuses, 20, $search)
            ], 200);
        } else {
            $search_paid = empty($search) ? [["orders.paid", 0]] : [["orders.paid", 0], $search];
            return $this->ReturnResponse(true, "Order list by order status", [
                'to_treat' => Order::getOrdersForSpaceByStatus($space->id, OrderStatus::whereIn('name', array('PAID', 'CONFIRMED'))->first()->id, $search),
                'to_deliver' => Order::getOrdersForSpaceToDeliver($space->id, OrderStatus::whereIn('name', array('PAID', 'CONFIRMED'))->first()->id, $serverId, $search),
                'history' => Order::getHistoryOrdersForSpace($space->id, $orderStatuses, 20, $search),
                'to_pay' => Order::getOrdersForSpaceByStatus($space->id, OrderStatus::where('name', 'CONFIRMED')->first()->id, $search_paid)
            ], 200);
        }
    }

    /*****
     * Generate online payment link for a order
     */
    public function generatePaymentOnlineLink(Request $request)
    {
        if (empty($request->order_number)) {
            return $this->ReturnResponse(false, "Order number is required", [], 404);
        }
        if (empty($request->return_url)) {
            return $this->ReturnResponse(false, "Order number is required", [], 404);
        }
        $return_url = $request->return_url;
        $order_number = $request->order_number;
        $order = Order::where('order_number', $order_number)->first();
        if (empty($order)) {
            return $this->ReturnResponse(false, "Order not found", [], 404);
        }
        $confirmedId = OrderStatus::where('name', 'CONFIRMED')->first()?->id;
        if ($order->order_status_id != $confirmedId) {
            return $this->ReturnResponse(false, "order being processed", [], 200);
        }
        $order->fees = 100 + $order->remaining_amount * 0.01;
        $order->save();
        /////creation de la transaction
        $trans_status = TransactionStatus::where('name', 'INIT')->first();
        $wallet = Wallet::where('name', 'CINETPAY')->first();
        $trans_type = TransactionType::where('name', "ONLINE")->first();
        $data = [
            "trans_status" => $trans_status->id,
            "wallet" => $wallet->id,
            "trans_type" => $trans_type->id,
            "request" => "",
            "response" => "",
            "callback_response" => "",
            "created_by" => $request->user()->id,
        ];
        $transaction = $this->createTransactionFromorder($order, $data);
        $data = [
            "amount"       => $order->remaining_amount + $order->fees,
            "description"   => "Paiement marchand",
            "libelle"   => "Paiement marchand",
            "transaction_id"   => $transaction->uuid,
            "name"   => $order->customer->lastname . '  ' . $order->customer->firstname,
            "tel"   => $order->customer->username,
            "return_url"   => $return_url,
        ];
        try {
            $payment = new CinetPayService();
            $urlPay = $payment->generatePaymentLink($data);
            if ($urlPay["code"] == '201') {
                $url = $urlPay["data"]["payment_url"];
                $data = [
                    "request" => $urlPay['request'],
                    "response" => $urlPay['response'],
                    "callback_response" => "",
                ];
                Transaction::where('uuid', $transaction->uuid)->update($data);
                return $this->ReturnResponse(true, "Payment link generated successfully", ["url" => $url], 200);
            } else {
                return $this->ReturnResponse(false, "An error has occurred", [], 200);
            }
        } catch (\Exception $e) {
            // $message = $e->getMessage();
            // $userid = $userRoles?->user?->id;
            // return $this->paymentFail($message, $devicesTokens);
        }
    }


    public function update(Request $request, $uuid): JsonResponse
    {
        $order = Order::where([
            ['uuid', $uuid],
            ['order_status_id', OrderStatus::where('name', 'CART')->first()->id]
        ])->first();
        if (!$order) {
            return $this->ReturnResponse(false, "Order not found", [], 404);
        }

        $validator = Validator::make(
            $request->all(),
            [
                'table_number'             => 'string',
                'rebate'                   => 'numeric|min:0',
            ]
        );

        //si erreur
        if ($validator->fails()) {
            return $this->ReturnResponse(false, $validator->errors(), [], 400);
        }

        try {
            DB::beginTransaction();
            $order->update([
                'table_number'          => $request->table_number
            ]);
            if (!is_null($request->rebate)) {
                $order->update(
                    [
                        'rebate'                => $request->rebate,
                        'remaining_amount'      => $order->amount_due - intval($request->rebate)
                    ]
                );
            }
            DB::commit();
            return $this->ReturnResponse(true, "Order Updated", []);
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossibme de mettre a jour la commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [], 500);
        }
    }

    public function updateSpace(Request $request, string $space_uuid, string $uuid): JsonResponse
    {
        return $this->update($request, $uuid);
    }


    public function updateOrderLineItem(Request $request, $uuid): JsonResponse
    {
        $order = Order::where([
            ['uuid', $uuid]
            // ,
            // ['order_status_id', OrderStatus::where('name', 'CART')->first()->id]
        ])->first();

        if (!$order) {
            return $this->ReturnResponse(false, "Order not found", [], 404);
        }

        $validator = Validator::make(
            $request->all(),
            [
                'space_id'                 => 'required|exists:spaces,id',
                'line_items'               => 'required|array',
                'line_items.*.stock_id'    => 'required|exists:stocks,id',
                'line_items.*.quantity'    => 'required|min:1',
            ]
        );
        //si erreur
        if ($validator->fails()) {
            return $this->ReturnResponse(false, $validator->errors(), [], 400);
        }


        $stock_id_list   = [];
        $list_line_items = [];

        foreach ($request->line_items as $key => $sline) {
            $stock_id_list[$request->line_items[$key]["stock_id"]] = $sline["quantity"]; // si deux element identique faire la somme
        }

        try {
            DB::beginTransaction();

            $stocks = Stock::where('space_id', $request->space_id)->whereIn('id', array_keys($stock_id_list))->get();

            if (count($stocks) != count($stock_id_list)) {
                return $this->ReturnResponse(false, "Certains articles n'appartiennent pas à cet espace", [], 404);
            }

            $order->lineItems()->delete();

            $amount_due = 0;


            foreach ($stocks as $key => $stock) {
                $line = [
                    "stock_id"      => $stock->id,
                    "order_id"      => $order->id,
                    "quantity"      => $stock_id_list[$stock->id],
                    "created_by"    => $request->user()->id,
                    "updated_by"    => $request->user()->id,
                ];

                $amount_due += $stock->price_after_sale * $stock_id_list[$stock->id];
                $list_line_items[] = $line;
            }

            LineItem::insert($list_line_items);

            $order->update([
                "amount_due"        => $amount_due,
                "remaining_amount"  => $amount_due - $order->rebate,
            ]);
            $order = Order::where('id', $order->id)->with(['orderStatus', 'lineItems'])->first();
            DB::commit();
            return $this->ReturnResponse(true, "La commande a été mise à jour", [$order->with('lineItems')]);
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossibme de mettre a jour la commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [], 500);
        }
    }


    public function updateOrderLineItemSpace(Request $request,  string $space_uuid, string $uuid): JsonResponse
    {
        //return $this->updateOrderLineItem($request, $uuid);

        $space = Space::Where([['uuid', $space_uuid], ['is_deleted', 0], ['is_active', 1]])->first();
        if (!$space) {
            return $this->ReturnResponse(false, "Space not found", [], 404);
        }
        $order = Order::select('orders.*') //
            ->join('order_statuses', 'order_statuses.id', '=', 'orders.order_status_id')
            ->join('line_items', 'line_items.order_id', '=', 'orders.id')
            ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
            ->with(['orderStatus', 'lineItems.stock.item'])
            ->where([['orders.is_deleted', 0], ['orders.is_active', 1]])
            ->where([
                ['stocks.space_id', $space->id],
                ['orders.uuid', $uuid],
                //['order_status_id', OrderStatus::where('name','!=','CART')->first()->id]
            ])->first();



        if (!$order) {
            return $this->ReturnResponse(false, "Order not found", [], 404);
        }

        $validator = Validator::make(
            $request->all(),
            [
                'space_id'                 => 'required|exists:spaces,id',
                'line_items'               => 'required|array',
                'line_items.*.stock_id'    => 'required|exists:stocks,id',
                'line_items.*.quantity'    => 'required|min:1',
            ]
        );
        //si erreur
        if ($validator->fails()) {
            return $this->ReturnResponse(false, $validator->errors(), [], 400);
        }


        $stock_id_list   = [];
        $list_line_items = [];

        foreach ($request->line_items as $key => $sline) {
            $stock_id_list[$request->line_items[$key]["stock_id"]] = $sline["quantity"]; // si deux element identique faire la somme
        }

        try {
            DB::beginTransaction();

            $stocks = Stock::where('space_id', $request->space_id)->whereIn('id', array_keys($stock_id_list))->get();

            if (count($stocks) != count($stock_id_list)) {
                return $this->ReturnResponse(false, "Certains articles n'appartiennent pas à cet espace", [], 404);
            }

            $order->lineItems()->delete();

            $amount_due = 0;


            foreach ($stocks as $key => $stock) {
                $line = [
                    "stock_id"      => $stock->id,
                    "order_id"      => $order->id,
                    "quantity"      => $stock_id_list[$stock->id],
                    "created_by"    => $request->user()->id,
                    "updated_by"    => $request->user()->id,
                ];

                $amount_due += $stock->price_after_sale * $stock_id_list[$stock->id];
                $list_line_items[] = $line;
            }

            LineItem::insert($list_line_items);

            $order->update([
                "amount_due"        => $amount_due,
                "remaining_amount"  => $amount_due - $order->rebate,
            ]);
            $order = Order::where('id', $order->id)->with(['orderStatus', 'lineItems'])->first();

            DB::commit();
            return $this->ReturnResponse(true, "La commande a été mise à jour", [$order->with('lineItems')]);
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossibme de mettre a jour la commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [], 500);
        }
    }



    public function pay(Request $request, string $uuid): JsonResponse
    {
        $data = [
            "id" => 1,
            "uuid" => Str::uuid(),
            "amount_paid" => 500,
        ];
        return $this->ReturnResponse(true, "Order Payed Successfully", $data);
    }

    public function deleteOrder(Request $request, string $uuid): JsonResponse
    {
        $order = Order::where([
            ['uuid', $uuid],
            ['is_deleted', 0],
            ['order_status_id', OrderStatus::where('name', 'CONFIRMED')->first()->id]
        ])->first();

        if (!$order) {
            return $this->ReturnResponse(false, "Order not found", [], 404);
        }

        try {
            DB::beginTransaction();
            $order->order_status_id = OrderStatus::where('name', 'CANCELED')->first()->id;
            $order->updated_at  = now();
            $order->updated_by  = $request->user()->id;

            $order->save();
            DB::commit();
            return $this->ReturnResponse(true, "Order Updated", [$order]);
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossibme de mettre a jour la commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [$order], 500);
        }

        return $this->ReturnResponse(true, "Order Deleted Successfully");
    }

    public function getBasketByUser(Request $request)
    {
        $orders = Order::where([
            ['created_by', $request->user()->id],
            ['order_status_id', OrderStatus::where('name', 'CART')->first()->id]
        ])->with(['lineItems'])->first();
        $basket = [];
        if ($orders) {
            foreach ($orders->lineItems as $line) {
                $stock = Stock::where('id', $line->stock_id)->with('item')->first();
                if (!empty($stock)) {
                    $p = array(
                        "count" => $line->quantity,
                        "description" => $stock->item->description,
                        "image" => $stock->item->img_1,
                        "is_prom" => $stock->item->is_on_promotion,
                        "name" => $stock->item->name,
                        "price" => $stock->price,
                        "quantity" => $stock->quantity,
                        "reduction" => $stock->off_by_number,
                        "reductionType" => 1,
                    );
                    array_push($basket, $p);
                }
            }
        }
        return $basket;
    }


    /**
     * payment Online Fail
     *
     * @param  mixed $message
     * @param  mixed $userid
     * @return array
     */
    private function paymentFail($message, $userid): array
    {
        if (request()->header('mobileId')) {
            // NotificationHelpers::send(
            //     request()->header('mobileId'),
            //     "Information sur la commande ",
            //     "Paiement Flex a échoué : " . $message
            // );
        }
        if (Device::where('user_id', $userid)->first()) {
            // NotificationHelpers::send(
            //     Device::where('user_id', $userid)->first()->device_token,
            //     "Information sur la commande ",
            //     "Paiement Flex a échoué : " . $message
            // );
        }
        return ['success' => false, "message" => $message, "status" => 4080];
    }
}
